#!/bin/bash
#
# Threaded Comments Migration Tool
# Interactive script for migrating threaded comments from Jira DC to Cloud
#
# Features:
#   1. Add Markers    - Adds migration markers to comments
#   2. Export JSONs   - Exports hierarchy and mapping data
#   3. Remove Markers - Removes migration markers from comments
#

set -e

# ============ CONFIGURATION ============
HIERARCHY_LIMIT=10000
MAPPING_LIMIT=1000
MARKER_BATCH_SIZE=100

# ============ COLORS ============
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
BOLD='\033[1m'
NC='\033[0m'

# ============ GLOBAL VARIABLES ============
JIRA_URL=""
USERNAME=""
PASSWORD=""

# ============ HELPER FUNCTIONS ============

print_header() {
    clear
    echo -e "${CYAN}"
    echo "╔════════════════════════════════════════════════════════════╗"
    echo "║                                                            ║"
    echo "║          THREADED COMMENTS MIGRATION TOOL                  ║"
    echo "║                                                            ║"
    echo "║          Migrate from Jira DC to Jira Cloud                ║"
    echo "║                                                            ║"
    echo "╚════════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
}

print_menu() {
    echo ""
    echo -e "${BOLD}Select an action:${NC}"
    echo ""
    echo -e "  ${GREEN}1)${NC} Add Markers      - Add migration markers to threaded comments"
    echo -e "  ${GREEN}2)${NC} Export JSONs     - Export hierarchy and issue mapping files"
    echo -e "  ${GREEN}3)${NC} Remove Markers   - Remove migration markers from comments"
    echo ""
    echo -e "  ${YELLOW}4)${NC} Exit"
    echo ""
}

log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

log_step() {
    echo -e "${CYAN}[STEP]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

# ============ AUTHENTICATION ============

prompt_credentials() {
    echo ""
    echo -e "${BOLD}Enter your Jira credentials:${NC}"
    echo ""

    # Jira URL
    read -p "Jira URL (e.g., https://jira.company.com): " JIRA_URL
    JIRA_URL="${JIRA_URL%/}"  # Remove trailing slash

    if [ -z "$JIRA_URL" ]; then
        log_error "Jira URL is required"
        return 1
    fi

    # Username
    echo ""
    read -p "Admin Username: " USERNAME

    if [ -z "$USERNAME" ]; then
        log_error "Username is required"
        return 1
    fi

    # Password (hidden input)
    echo ""
    read -s -p "Admin Password: " PASSWORD
    echo ""

    if [ -z "$PASSWORD" ]; then
        log_error "Password is required"
        return 1
    fi

    return 0
}

validate_credentials() {
    echo ""
    log_step "Validating credentials..."

    # Test authentication
    response=$(curl -s -w "\n%{http_code}" \
        -u "$USERNAME:$PASSWORD" \
        "$JIRA_URL/rest/api/2/myself" 2>/dev/null)

    http_code=$(echo "$response" | tail -n1)
    body=$(echo "$response" | sed '$d')

    if [ "$http_code" = "401" ]; then
        log_error "Authentication failed. Please check your username and password."
        return 1
    fi

    if [ "$http_code" = "403" ]; then
        log_error "Access forbidden. Your account may be locked or disabled."
        return 1
    fi

    if [ "$http_code" != "200" ]; then
        log_error "Cannot connect to Jira (HTTP $http_code). Please check the URL."
        return 1
    fi

    # Check admin permissions
    log_step "Checking administrator permissions..."

    admin_check=$(curl -s -w "\n%{http_code}" \
        -u "$USERNAME:$PASSWORD" \
        "$JIRA_URL/rest/threaded-comments/1.0/migration/jobs" 2>/dev/null)

    admin_http_code=$(echo "$admin_check" | tail -n1)

    if [ "$admin_http_code" = "403" ]; then
        log_error "Permission denied. You must be a Jira System Administrator."
        log_error "Please contact your Jira admin to grant you the required permissions."
        return 1
    fi

    if [ "$admin_http_code" != "200" ]; then
        log_error "Failed to verify admin permissions (HTTP $admin_http_code)."
        log_error "Make sure the Threaded Comments plugin is installed."
        return 1
    fi

    display_name=$(echo "$body" | grep -o '"displayName":"[^"]*"' | cut -d'"' -f4)
    log_success "Authenticated as: $display_name"
    log_success "Administrator permissions verified"

    return 0
}

# ============ ADD MARKERS ============

do_add_markers() {
    echo ""
    echo -e "${BOLD}═══════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}                     ADD MARKERS                           ${NC}"
    echo -e "${BOLD}═══════════════════════════════════════════════════════════${NC}"
    echo ""
    echo "This will add migration markers to all threaded comments."
    echo "Markers format: :::VOTAZZ_MIG:commentId:::"
    echo ""
    echo "These markers are used to identify comments during migration"
    echo "from Jira DC to Jira Cloud."
    echo ""

    # Get count first
    log_step "Counting comments to process..."

    response=$(curl -s -u "$USERNAME:$PASSWORD" \
        "$JIRA_URL/rest/threaded-comments/1.0/migration/export-data?offset=0&limit=1")

    total_count=$(echo "$response" | grep -o '"totalCount":[0-9]*' | cut -d':' -f2)

    if [ -z "$total_count" ] || [ "$total_count" = "0" ]; then
        log_warn "No threaded comments found to process."
        return 0
    fi

    echo -e "Found ${BOLD}$total_count${NC} comment records to process."
    echo ""
    read -p "Do you want to proceed? (yes/no): " confirm

    if [ "$confirm" != "yes" ]; then
        log_warn "Operation cancelled."
        return 0
    fi

    echo ""
    log_step "Starting marker addition..."

    # Call the async endpoint
    response=$(curl -s -w "\n%{http_code}" -X POST \
        -u "$USERNAME:$PASSWORD" \
        "$JIRA_URL/rest/threaded-comments/1.0/migration/add-markers")

    http_code=$(echo "$response" | tail -n1)
    body=$(echo "$response" | sed '$d')

    if [ "$http_code" = "409" ]; then
        existing_job=$(echo "$body" | grep -o '"existingJobId":"[^"]*"' | cut -d'"' -f4)
        log_warn "A job is already running: $existing_job"
        log_info "Use the status check to monitor progress."
        return 0
    fi

    if [ "$http_code" != "200" ]; then
        log_error "Failed to start marker addition (HTTP $http_code)"
        echo "$body"
        return 1
    fi

    job_id=$(echo "$body" | grep -o '"jobId":"[^"]*"' | cut -d'"' -f4)
    log_success "Job started: $job_id"
    echo ""

    # Monitor progress
    monitor_job "$job_id" "Adding markers"
}

# ============ REMOVE MARKERS ============

do_remove_markers() {
    echo ""
    echo -e "${BOLD}═══════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}                   REMOVE MARKERS                          ${NC}"
    echo -e "${BOLD}═══════════════════════════════════════════════════════════${NC}"
    echo ""
    echo "This will remove all migration markers from comments."
    echo "Use this after migration is complete, or to clean up."
    echo ""
    echo -e "${YELLOW}Warning: Only run this after you have verified the migration${NC}"
    echo -e "${YELLOW}was successful in Jira Cloud.${NC}"
    echo ""

    read -p "Do you want to proceed? (yes/no): " confirm

    if [ "$confirm" != "yes" ]; then
        log_warn "Operation cancelled."
        return 0
    fi

    echo ""
    log_step "Starting marker removal..."

    response=$(curl -s -w "\n%{http_code}" -X POST \
        -u "$USERNAME:$PASSWORD" \
        "$JIRA_URL/rest/threaded-comments/1.0/migration/remove-markers")

    http_code=$(echo "$response" | tail -n1)
    body=$(echo "$response" | sed '$d')

    if [ "$http_code" = "409" ]; then
        existing_job=$(echo "$body" | grep -o '"existingJobId":"[^"]*"' | cut -d'"' -f4)
        log_warn "A job is already running: $existing_job"
        return 0
    fi

    if [ "$http_code" != "200" ]; then
        log_error "Failed to start marker removal (HTTP $http_code)"
        echo "$body"
        return 1
    fi

    job_id=$(echo "$body" | grep -o '"jobId":"[^"]*"' | cut -d'"' -f4)
    log_success "Job started: $job_id"
    echo ""

    monitor_job "$job_id" "Removing markers"
}

# ============ MONITOR JOB ============

monitor_job() {
    local job_id=$1
    local action_name=$2

    log_step "Monitoring job progress..."
    echo ""

    while true; do
        response=$(curl -s -u "$USERNAME:$PASSWORD" \
            "$JIRA_URL/rest/threaded-comments/1.0/migration/status/$job_id")

        status=$(echo "$response" | grep -o '"status":"[^"]*"' | cut -d'"' -f4)
        processed=$(echo "$response" | grep -o '"processedCount":[0-9]*' | cut -d':' -f2)
        total=$(echo "$response" | grep -o '"totalCount":[0-9]*' | cut -d':' -f2)

        if [ -n "$total" ] && [ "$total" -gt 0 ]; then
            percent=$((processed * 100 / total))
            echo -ne "\r  $action_name: $processed / $total ($percent%) - Status: $status    "
        fi

        if [ "$status" = "COMPLETED" ]; then
            echo ""
            echo ""
            log_success "$action_name completed successfully!"
            message=$(echo "$response" | grep -o '"message":"[^"]*"' | cut -d'"' -f4)
            if [ -n "$message" ]; then
                log_info "$message"
            fi
            break
        fi

        if [ "$status" = "FAILED" ]; then
            echo ""
            echo ""
            log_error "$action_name failed!"
            message=$(echo "$response" | grep -o '"message":"[^"]*"' | cut -d'"' -f4)
            if [ -n "$message" ]; then
                log_error "$message"
            fi
            break
        fi

        if [ "$status" = "CANCELLED" ]; then
            echo ""
            echo ""
            log_warn "$action_name was cancelled."
            break
        fi

        sleep 2
    done
}

# ============ EXPORT JSONs ============

do_export_jsons() {
    echo ""
    echo -e "${BOLD}═══════════════════════════════════════════════════════════${NC}"
    echo -e "${BOLD}                    EXPORT JSONs                           ${NC}"
    echo -e "${BOLD}═══════════════════════════════════════════════════════════${NC}"
    echo ""
    echo "This will export two JSON files:"
    echo ""
    echo "  1. comments_hierarchy_export.json"
    echo "     Contains comment parent-child relationships"
    echo "     Format: [{issueId, commentId, parentCommentId}, ...]"
    echo ""
    echo "  2. issues_mapping_export.json"
    echo "     Contains issue ID to issue key mapping"
    echo "     Format: {\"issueId\": \"PROJECT-123\", ...}"
    echo ""

    read -p "Do you want to proceed? (yes/no): " confirm

    if [ "$confirm" != "yes" ]; then
        log_warn "Operation cancelled."
        return 0
    fi

    echo ""
    export_hierarchy
    echo ""
    export_mapping

    echo ""
    log_success "Export completed!"
    echo ""
    echo "Files created in: $(pwd)"
    echo "  - comments_hierarchy_export.json"
    echo "  - issues_mapping_export.json"
}

export_hierarchy() {
    local output_file="comments_hierarchy_export.json"
    local temp_file=$(mktemp)
    local offset=0
    local total=0
    local has_more="true"
    local first_chunk="true"

    log_step "Exporting comment hierarchy..."

    # Start JSON array
    printf '[' > "$temp_file"

    while [ "$has_more" = "true" ]; do
        response=$(curl -s -u "$USERNAME:$PASSWORD" \
            "$JIRA_URL/rest/threaded-comments/1.0/migration/export-data?offset=$offset&limit=$HIERARCHY_LIMIT")

        error=$(echo "$response" | jq -r '.error // empty' 2>/dev/null)
        if [ -n "$error" ]; then
            log_error "API error: $error"
            rm -f "$temp_file"
            return 1
        fi

        total=$(echo "$response" | jq -r '.totalCount // 0')
        returned=$(echo "$response" | jq -r '.returnedCount // 0')
        has_more=$(echo "$response" | jq -r '.hasMore // false')

        if [ "$returned" -gt 0 ]; then
            # Extract data array content using jq (removes [ and ] from array)
            data_content=$(echo "$response" | jq -c '.data[]' 2>/dev/null | paste -sd ',' -)

            if [ -n "$data_content" ]; then
                if [ "$first_chunk" = "true" ]; then
                    first_chunk="false"
                else
                    printf ',' >> "$temp_file"
                fi
                printf '%s' "$data_content" >> "$temp_file"
            fi
        fi

        processed=$((offset + returned))
        if [ -n "$total" ] && [ "$total" -gt 0 ]; then
            percent=$((processed * 100 / total))
            echo -ne "\r  Progress: $processed / $total ($percent%)    "
        fi

        offset=$((offset + HIERARCHY_LIMIT))

        [ "$has_more" != "true" ] && has_more="false"
    done

    # Close JSON array
    printf ']' >> "$temp_file"

    # Format JSON
    jq '.' "$temp_file" > "$output_file" 2>/dev/null || mv "$temp_file" "$output_file"
    rm -f "$temp_file" 2>/dev/null

    echo ""
    log_info "Hierarchy exported: $output_file ($total records)"
}

export_mapping() {
    local output_file="issues_mapping_export.json"
    local temp_file=$(mktemp)
    local offset=0
    local total=0
    local has_more="true"
    local first_chunk="true"

    log_step "Exporting issue mapping..."

    # Start JSON object
    printf '{' > "$temp_file"

    while [ "$has_more" = "true" ]; do
        response=$(curl -s -u "$USERNAME:$PASSWORD" \
            "$JIRA_URL/rest/threaded-comments/1.0/migration/export-mapping-data?offset=$offset&limit=$MAPPING_LIMIT")

        error=$(echo "$response" | jq -r '.error // empty' 2>/dev/null)
        if [ -n "$error" ]; then
            log_error "API error: $error"
            rm -f "$temp_file"
            return 1
        fi

        total=$(echo "$response" | jq -r '.totalCount // 0')
        returned=$(echo "$response" | jq -r '.returnedCount // 0')
        has_more=$(echo "$response" | jq -r '.hasMore // false')

        if [ "$returned" -gt 0 ]; then
            # Extract data object entries using jq
            data_content=$(echo "$response" | jq -r '.data | to_entries | map("\"\(.key)\":\"\(.value)\"") | join(",")' 2>/dev/null)

            if [ -n "$data_content" ]; then
                if [ "$first_chunk" = "true" ]; then
                    first_chunk="false"
                else
                    printf ',' >> "$temp_file"
                fi
                printf '%s' "$data_content" >> "$temp_file"
            fi
        fi

        processed=$((offset + returned))
        if [ -n "$total" ] && [ "$total" -gt 0 ]; then
            percent=$((processed * 100 / total))
            echo -ne "\r  Progress: $processed / $total ($percent%)    "
        fi

        offset=$((offset + MAPPING_LIMIT))

        [ "$has_more" != "true" ] && has_more="false"
    done

    # Close JSON object
    printf '}' >> "$temp_file"

    # Format JSON
    jq '.' "$temp_file" > "$output_file" 2>/dev/null || mv "$temp_file" "$output_file"
    rm -f "$temp_file" 2>/dev/null

    echo ""
    log_info "Mapping exported: $output_file ($total issues)"
}

# ============ MAIN ============

main() {
    print_header

    # Prompt for credentials
    if ! prompt_credentials; then
        echo ""
        read -p "Press Enter to continue..."
        main
        return
    fi

    # Validate credentials
    if ! validate_credentials; then
        echo ""
        read -p "Press Enter to try again..."
        main
        return
    fi

    # Main menu loop
    while true; do
        echo ""
        print_menu

        read -p "Enter your choice (1-4): " choice

        case $choice in
            1)
                do_add_markers
                echo ""
                read -p "Press Enter to continue..."
                ;;
            2)
                do_export_jsons
                echo ""
                read -p "Press Enter to continue..."
                ;;
            3)
                do_remove_markers
                echo ""
                read -p "Press Enter to continue..."
                ;;
            4)
                echo ""
                log_info "Goodbye!"
                exit 0
                ;;
            *)
                log_warn "Invalid choice. Please enter 1, 2, 3, or 4."
                ;;
        esac
    done
}

# Run main
main
